<?php
// cookies_api.php - API serveur pour DEC Cookies Groupe 4
// Groupe 4 Sécurité

// ===== CONFIGURATION =====
$CONFIG = [
    'API_KEY' => 'G4_SECURITY_2024_API_KEY_CHANGE_ME',
    'GROUP_ID' => 'G4',
    'STORAGE_DIR' => __DIR__ . '/cookies_data/',
    'MAX_FILE_SIZE' => 5242880, // 5MB
    'ALLOWED_ORIGINS' => ['chrome-extension://*'],
    'ENABLE_ENCRYPTION' => false,
    'ENCRYPTION_KEY' => 'G4_ENCRYPTION_KEY_CHANGE_ME'
];

// ===== INITIALISATION =====
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: X-API-Key, X-Group-ID, Content-Type');
header('Access-Control-Max-Age: 86400'); // 24 heures

// Gérer les requêtes OPTIONS (CORS preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Initialiser le système de fichiers
initStorageSystem($CONFIG['STORAGE_DIR']);

// ===== ROUTEUR =====
$request_method = $_SERVER['REQUEST_METHOD'];
$request_path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);

try {
    // Route: Test de connexion
    if ($request_path === '/cookies_api.php/test' && $request_method === 'GET') {
        handleTestEndpoint($CONFIG);
    }
    
    // Route: Upload de session
    elseif ($request_path === '/cookies_api.php/upload' && $request_method === 'POST') {
        handleUploadEndpoint($CONFIG);
    }
    
    // Route: Download de sessions
    elseif ($request_path === '/cookies_api.php/download' && $request_method === 'GET') {
        handleDownloadEndpoint($CONFIG);
    }
    
    // Route: Statistiques
    elseif ($request_path === '/cookies_api.php/stats' && $request_method === 'GET') {
        handleStatsEndpoint($CONFIG);
    }
    
    // Route: List sessions
    elseif ($request_path === '/cookies_api.php/list' && $request_method === 'GET') {
        handleListEndpoint($CONFIG);
    }
    
    // Route: Delete session
    elseif ($request_path === '/cookies_api.php/delete' && $request_method === 'POST') {
        handleDeleteEndpoint($CONFIG);
    }
    
    // Route: Backup
    elseif ($request_path === '/cookies_api.php/backup' && $request_method === 'GET') {
        handleBackupEndpoint($CONFIG);
    }
    
    // Route non trouvée
    else {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'Endpoint non trouvé',
            'available_endpoints' => [
                'GET /cookies_api.php/test',
                'POST /cookies_api.php/upload',
                'GET /cookies_api.php/download',
                'GET /cookies_api.php/stats',
                'GET /cookies_api.php/list',
                'POST /cookies_api.php/delete',
                'GET /cookies_api.php/backup'
            ]
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erreur serveur: ' . $e->getMessage(),
        'trace' => $CONFIG['DEBUG'] ? $e->getTraceAsString() : null
    ]);
}

// ===== FONCTIONS D'INITIALISATION =====
function initStorageSystem($storageDir) {
    // Créer le répertoire principal
    if (!file_exists($storageDir)) {
        if (!mkdir($storageDir, 0755, true)) {
            throw new Exception("Impossible de créer le répertoire de stockage: $storageDir");
        }
    }
    
    // Créer les sous-répertoires
    $subdirs = ['sessions', 'backups', 'logs', 'temp'];
    foreach ($subdirs as $subdir) {
        $dir = $storageDir . $subdir . '/';
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
        }
    }
    
    // Créer l'index s'il n'existe pas
    $indexFile = $storageDir . 'index.json';
    if (!file_exists($indexFile)) {
        file_put_contents($indexFile, json_encode([
            'version' => '1.0',
            'created' => date('c'),
            'total_sessions' => 0,
            'last_updated' => date('c'),
            'sessions' => []
        ], JSON_PRETTY_PRINT));
    }
    
    // Créer le fichier de logs s'il n'existe pas
    $logFile = $storageDir . 'logs/api.log';
    if (!file_exists(dirname($logFile))) {
        mkdir(dirname($logFile), 0755, true);
    }
}

// ===== FONCTIONS D'AUTHENTIFICATION =====
function authenticateRequest($config) {
    $headers = getallheaders();
    
    // Vérifier la clé API
    if (!isset($headers['X-API-Key']) || $headers['X-API-Key'] !== $config['API_KEY']) {
        logMessage($config, 'AUTH_FAIL', 'Clé API invalide');
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Clé API invalide']);
        exit;
    }
    
    // Vérifier le groupe ID
    if (!isset($headers['X-Group-ID']) || $headers['X-Group-ID'] !== $config['GROUP_ID']) {
        logMessage($config, 'AUTH_FAIL', 'Groupe non autorisé: ' . ($headers['X-Group-ID'] ?? 'none'));
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'Groupe non autorisé']);
        exit;
    }
    
    // Authentification réussie
    logMessage($config, 'AUTH_SUCCESS', 'Authentification réussie pour le groupe G4');
    return true;
}

// ===== GESTION DES ENDPOINTS =====
function handleTestEndpoint($config) {
    authenticateRequest($config);
    
    // Récupérer les infos système
    $storageDir = $config['STORAGE_DIR'];
    $indexFile = $storageDir . 'index.json';
    $index = file_exists($indexFile) ? json_decode(file_get_contents($indexFile), true) : [];
    
    echo json_encode([
        'success' => true,
        'status' => 'online',
        'server' => [
            'name' => 'DEC Cookies API G4',
            'version' => '1.0',
            'group' => $config['GROUP_ID'],
            'timestamp' => date('c'),
            'php_version' => PHP_VERSION,
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'
        ],
        'storage' => [
            'total_sessions' => $index['total_sessions'] ?? 0,
            'last_updated' => $index['last_updated'] ?? date('c'),
            'free_space' => disk_free_space($storageDir),
            'total_space' => disk_total_space($storageDir),
            'directory' => $storageDir
        ],
        'message' => 'API opérationnelle - Groupe 4 Sécurité'
    ]);
}

function handleUploadEndpoint($config) {
    authenticateRequest($config);
    
    // Lire les données JSON
    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Données JSON invalides ou vides']);
        return;
    }
    
    // Valider les données requises
    $required_fields = ['id', 'name', 'domain', 'cookies', 'timestamp'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => "Champ requis manquant: $field"]);
            return;
        }
    }
    
    // Préparer les données de session
    $session_id = $input['id'];
    $session_data = [
        'session' => $input,
        'metadata' => [
            'uploaded_at' => date('c'),
            'upload_ip' => $_SERVER['REMOTE_ADDR'],
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'file_size' => strlen(json_encode($input)),
            'group_id' => $config['GROUP_ID']
        ]
    ];
    
    // Sauvegarder la session dans un fichier
    $session_dir = $config['STORAGE_DIR'] . 'sessions/';
    $session_file = $session_dir . $session_id . '.json';
    
    if (file_put_contents($session_file, json_encode($session_data, JSON_PRETTY_PRINT))) {
        // Mettre à jour l'index
        updateIndex($config, $session_id, $input);
        
        logMessage($config, 'UPLOAD_SUCCESS', "Session {$input['name']} sauvegardée (ID: $session_id)");
        
        echo json_encode([
            'success' => true,
            'session_id' => $session_id,
            'sync_id' => uniqid('sync_', true),
            'timestamp' => date('c'),
            'message' => 'Session sauvegardée avec succès',
            'details' => [
                'name' => $input['name'],
                'domain' => $input['domain'],
                'cookies_count' => count($input['cookies']),
                'file_size' => filesize($session_file)
            ]
        ]);
    } else {
        http_response_code(500);
        logMessage($config, 'UPLOAD_ERROR', "Erreur sauvegarde session $session_id");
        echo json_encode(['success' => false, 'error' => 'Erreur lors de la sauvegarde']);
    }
}

function handleDownloadEndpoint($config) {
    authenticateRequest($config);
    
    // Lire l'index
    $indexFile = $config['STORAGE_DIR'] . 'index.json';
    if (!file_exists($indexFile)) {
        echo json_encode([]);
        return;
    }
    
    $index = json_decode(file_get_contents($indexFile), true);
    $sessions = $index['sessions'] ?? [];
    
    // Filtrer par groupe si spécifié
    $groupId = $_GET['group'] ?? $config['GROUP_ID'];
    $filtered_sessions = array_filter($sessions, function($session) use ($groupId) {
        return ($session['group_id'] ?? $config['GROUP_ID']) === $groupId;
    });
    
    // Charger les données complètes des sessions
    $result = [];
    $session_dir = $config['STORAGE_DIR'] . 'sessions/';
    
    foreach ($filtered_sessions as $session_id => $session_info) {
        $session_file = $session_dir . $session_id . '.json';
        if (file_exists($session_file)) {
            $session_data = json_decode(file_get_contents($session_file), true);
            if ($session_data) {
                $result[] = array_merge(
                    $session_data['session'],
                    [
                        'server_info' => [
                            'stored_at' => $session_data['metadata']['uploaded_at'],
                            'file_size' => $session_data['metadata']['file_size']
                        ]
                    ]
                );
            }
        }
    }
    
    // Limiter le nombre de résultats si spécifié
    $limit = intval($_GET['limit'] ?? 50);
    if ($limit > 0 && count($result) > $limit) {
        $result = array_slice($result, 0, $limit);
    }
    
    echo json_encode([
        'success' => true,
        'count' => count($result),
        'total' => count($filtered_sessions),
        'timestamp' => date('c'),
        'sessions' => $result
    ]);
}

function handleStatsEndpoint($config) {
    authenticateRequest($config);
    
    $indexFile = $config['STORAGE_DIR'] . 'index.json';
    if (!file_exists($indexFile)) {
        echo json_encode([
            'success' => true,
            'stats' => [
                'total_sessions' => 0,
                'total_cookies' => 0,
                'total_size' => 0,
                'domains' => [],
                'last_updated' => date('c')
            ]
        ]);
        return;
    }
    
    $index = json_decode(file_get_contents($indexFile), true);
    $sessions = $index['sessions'] ?? [];
    
    // Calculer les statistiques
    $stats = [
        'total_sessions' => count($sessions),
        'total_cookies' => 0,
        'total_size' => 0,
        'domains' => [],
        'groups' => [],
        'timeline' => []
    ];
    
    $session_dir = $config['STORAGE_DIR'] . 'sessions/';
    
    foreach ($sessions as $session_id => $session_info) {
        // Compter les cookies
        $stats['total_cookies'] += $session_info['cookie_count'] ?? 0;
        
        // Taille du fichier
        $session_file = $session_dir . $session_id . '.json';
        if (file_exists($session_file)) {
            $stats['total_size'] += filesize($session_file);
        }
        
        // Domaines
        $domain = $session_info['domain'] ?? 'unknown';
        if (!isset($stats['domains'][$domain])) {
            $stats['domains'][$domain] = 0;
        }
        $stats['domains'][$domain]++;
        
        // Groupes
        $group = $session_info['group_id'] ?? 'unknown';
        if (!isset($stats['groups'][$group])) {
            $stats['groups'][$group] = 0;
        }
        $stats['groups'][$group]++;
        
        // Timeline (par jour)
        $date = date('Y-m-d', strtotime($session_info['timestamp'] ?? 'now'));
        if (!isset($stats['timeline'][$date])) {
            $stats['timeline'][$date] = 0;
        }
        $stats['timeline'][$date]++;
    }
    
    // Trier les domaines par fréquence
    arsort($stats['domains']);
    $stats['domains'] = array_slice($stats['domains'], 0, 10);
    
    // Formater la taille
    $stats['total_size_formatted'] = formatBytes($stats['total_size']);
    
    echo json_encode([
        'success' => true,
        'stats' => $stats,
        'storage_info' => [
            'free_space' => disk_free_space($config['STORAGE_DIR']),
            'total_space' => disk_total_space($config['STORAGE_DIR']),
            'used_space' => $stats['total_size']
        ],
        'timestamp' => date('c')
    ]);
}

function handleListEndpoint($config) {
    authenticateRequest($config);
    
    $indexFile = $config['STORAGE_DIR'] . 'index.json';
    if (!file_exists($indexFile)) {
        echo json_encode([
            'success' => true,
            'sessions' => [],
            'count' => 0
        ]);
        return;
    }
    
    $index = json_decode(file_get_contents($indexFile), true);
    $sessions = $index['sessions'] ?? [];
    
    // Préparer la liste
    $session_list = [];
    foreach ($sessions as $session_id => $session_info) {
        $session_list[] = [
            'id' => $session_id,
            'name' => $session_info['name'] ?? 'Unnamed',
            'domain' => $session_info['domain'] ?? 'unknown',
            'group_id' => $session_info['group_id'] ?? $config['GROUP_ID'],
            'timestamp' => $session_info['timestamp'] ?? date('c'),
            'cookie_count' => $session_info['cookie_count'] ?? 0,
            'size' => $session_info['file_size'] ?? 0,
            'size_formatted' => formatBytes($session_info['file_size'] ?? 0)
        ];
    }
    
    // Trier par date (plus récent d'abord)
    usort($session_list, function($a, $b) {
        return strtotime($b['timestamp']) - strtotime($a['timestamp']);
    });
    
    echo json_encode([
        'success' => true,
        'count' => count($session_list),
        'sessions' => $session_list,
        'timestamp' => date('c')
    ]);
}

function handleDeleteEndpoint($config) {
    authenticateRequest($config);
    
    $input = json_decode(file_get_contents('php://input'), true);
    $session_id = $input['session_id'] ?? null;
    
    if (!$session_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Session ID requis']);
        return;
    }
    
    // Supprimer le fichier de session
    $session_file = $config['STORAGE_DIR'] . 'sessions/' . $session_id . '.json';
    $deleted = false;
    
    if (file_exists($session_file)) {
        $deleted = unlink($session_file);
    }
    
    // Mettre à jour l'index
    if ($deleted) {
        $indexFile = $config['STORAGE_DIR'] . 'index.json';
        $index = json_decode(file_get_contents($indexFile), true);
        
        if (isset($index['sessions'][$session_id])) {
            unset($index['sessions'][$session_id]);
            $index['total_sessions'] = count($index['sessions']);
            $index['last_updated'] = date('c');
            
            file_put_contents($indexFile, json_encode($index, JSON_PRETTY_PRINT));
        }
        
        logMessage($config, 'DELETE_SUCCESS', "Session $session_id supprimée");
        
        echo json_encode([
            'success' => true,
            'message' => 'Session supprimée avec succès',
            'session_id' => $session_id,
            'timestamp' => date('c')
        ]);
    } else {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'Session non trouvée ou déjà supprimée',
            'session_id' => $session_id
        ]);
    }
}

function handleBackupEndpoint($config) {
    authenticateRequest($config);
    
    // Créer un fichier de backup ZIP
    $backup_dir = $config['STORAGE_DIR'] . 'backups/';
    $backup_file = $backup_dir . 'backup_' . date('Y-m-d_H-i-s') . '.zip';
    
    // Créer une archive ZIP
    $zip = new ZipArchive();
    if ($zip->open($backup_file, ZipArchive::CREATE) === TRUE) {
        // Ajouter tous les fichiers de sessions
        $session_dir = $config['STORAGE_DIR'] . 'sessions/';
        $session_files = glob($session_dir . '*.json');
        
        foreach ($session_files as $file) {
            $zip->addFile($file, 'sessions/' . basename($file));
        }
        
        // Ajouter l'index
        $index_file = $config['STORAGE_DIR'] . 'index.json';
        if (file_exists($index_file)) {
            $zip->addFile($index_file, 'index.json');
        }
        
        // Ajouter un fichier info
        $info = [
            'backup_date' => date('c'),
            'total_sessions' => count($session_files),
            'group_id' => $config['GROUP_ID'],
            'server' => $_SERVER['SERVER_NAME'] ?? 'localhost'
        ];
        
        $zip->addFromString('info.json', json_encode($info, JSON_PRETTY_PRINT));
        $zip->close();
        
        // Retourner le lien de téléchargement
        $download_url = str_replace($_SERVER['DOCUMENT_ROOT'], '', $backup_file);
        
        echo json_encode([
            'success' => true,
            'backup_file' => basename($backup_file),
            'download_url' => $download_url,
            'file_size' => filesize($backup_file),
            'file_size_formatted' => formatBytes(filesize($backup_file)),
            'timestamp' => date('c'),
            'message' => 'Backup créé avec succès'
        ]);
        
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Impossible de créer le fichier de backup'
        ]);
    }
}

// ===== FONCTIONS UTILITAIRES =====
function updateIndex($config, $session_id, $session_data) {
    $indexFile = $config['STORAGE_DIR'] . 'index.json';
    $index = file_exists($indexFile) ? json_decode(file_get_contents($indexFile), true) : [
        'version' => '1.0',
        'created' => date('c'),
        'total_sessions' => 0,
        'last_updated' => date('c'),
        'sessions' => []
    ];
    
    // Ajouter ou mettre à jour la session dans l'index
    $index['sessions'][$session_id] = [
        'name' => $session_data['name'] ?? 'Unnamed',
        'domain' => $session_data['domain'] ?? 'unknown',
        'group_id' => $session_data['groupId'] ?? $config['GROUP_ID'],
        'timestamp' => $session_data['timestamp'] ?? date('c'),
        'cookie_count' => count($session_data['cookies'] ?? []),
        'file_size' => strlen(json_encode($session_data)),
        'last_updated' => date('c')
    ];
    
    $index['total_sessions'] = count($index['sessions']);
    $index['last_updated'] = date('c');
    
    file_put_contents($indexFile, json_encode($index, JSON_PRETTY_PRINT));
}

function logMessage($config, $type, $message) {
    $log_dir = $config['STORAGE_DIR'] . 'logs/';
    $log_file = $log_dir . 'api_' . date('Y-m-d') . '.log';
    
    $log_entry = sprintf(
        "[%s] [%s] %s - IP: %s - UA: %s\n",
        date('Y-m-d H:i:s'),
        $type,
        $message,
        $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        substr($_SERVER['HTTP_USER_AGENT'] ?? 'unknown', 0, 100)
    );
    
    file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
}

function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    
    return round($bytes, $precision) . ' ' . $units[$pow];
}

// Fonction de déchiffrement (optionnelle)
function decryptData($encrypted_data, $key) {
    // Implémentation basique - à améliorer pour la production
    $method = 'AES-256-CBC';
    $iv_length = openssl_cipher_iv_length($method);
    
    $data = base64_decode($encrypted_data);
    $iv = substr($data, 0, $iv_length);
    $encrypted = substr($data, $iv_length);
    
    return openssl_decrypt($encrypted, $method, $key, 0, $iv);
}