<?php
// rate_limiter.php - Système de limitation de débit
// Groupe 4 Sécurité

class RateLimiter {
    private $config;
    private $storageDir;
    
    public function __construct($config) {
        $this->config = $config;
        $this->storageDir = $config['STORAGE_DIR'] . 'temp/rate_limit/';
        
        if (!is_dir($this->storageDir)) {
            mkdir($this->storageDir, 0755, true);
        }
    }
    
    public function checkLimit($ip) {
        $limit = $this->config['RATE_LIMIT_REQUESTS'] ?? 100;
        $period = $this->config['RATE_LIMIT_PERIOD'] ?? 60;
        
        $ipFile = $this->storageDir . md5($ip) . '.json';
        
        if (file_exists($ipFile)) {
            $data = json_decode(file_get_contents($ipFile), true);
            
            // Vérifier si la période est écoulée
            if (time() - $data['first_request'] > $period) {
                // Réinitialiser le compteur
                $data = [
                    'count' => 1,
                    'first_request' => time(),
                    'last_request' => time()
                ];
            } else {
                // Vérifier la limite
                if ($data['count'] >= $limit) {
                    return false;
                }
                
                // Incrémenter le compteur
                $data['count']++;
                $data['last_request'] = time();
            }
        } else {
            // Première requête
            $data = [
                'count' => 1,
                'first_request' => time(),
                'last_request' => time()
            ];
        }
        
        // Sauvegarder les données
        file_put_contents($ipFile, json_encode($data));
        
        // Nettoyer les vieux fichiers
        $this->cleanupOldFiles($period);
        
        return true;
    }
    
    private function cleanupOldFiles($period) {
        $files = glob($this->storageDir . '*.json');
        $cutoff = time() - ($period * 2); // Double de la période
        
        foreach ($files as $file) {
            if (filemtime($file) < $cutoff) {
                unlink($file);
            }
        }
    }
    
    public function getRemainingRequests($ip) {
        $limit = $this->config['RATE_LIMIT_REQUESTS'] ?? 100;
        $period = $this->config['RATE_LIMIT_PERIOD'] ?? 60;
        
        $ipFile = $this->storageDir . md5($ip) . '.json';
        
        if (!file_exists($ipFile)) {
            return $limit;
        }
        
        $data = json_decode(file_get_contents($ipFile), true);
        
        // Vérifier si la période est écoulée
        if (time() - $data['first_request'] > $period) {
            return $limit;
        }
        
        return max(0, $limit - $data['count']);
    }
}
?>