<?php
// cookies_api.php - API serveur pour DEC Cookies Groupe 4
// Groupe 4 Sécurité

// ===== CHARGEMENT DE LA CONFIGURATION =====
$CONFIG_FILE = __DIR__ . '/config.php';
if (!file_exists($CONFIG_FILE)) {
    // Configuration par défaut si config.php n'existe pas
    $CONFIG = [
        'API_KEY' => 'G4_SECURITY_KEY_' . date('Y'),
        'GROUP_ID' => 'G4',
        'STORAGE_DIR' => __DIR__ . '/data/',
        'MAX_FILE_SIZE' => 5242880,
        'ENABLE_LOGS' => true,
        'REQUIRE_HTTPS' => false
    ];
} else {
    $CONFIG = require $CONFIG_FILE;
}

// ===== VÉRIFICATION MAINTENANCE =====
if ($CONFIG['MAINTENANCE_MODE'] ?? false) {
    http_response_code(503);
    header('Retry-After: 3600');
    echo json_encode([
        'error' => 'API en maintenance',
        'message' => $CONFIG['MAINTENANCE_MESSAGE'] ?? 'Veuillez réessayer plus tard.',
        'timestamp' => date('c')
    ]);
    exit;
}

// ===== INITIALISATION =====
header('Content-Type: application/json; charset=utf-8');
header('X-API-Version: ' . ($CONFIG['API_VERSION'] ?? '1.0.0'));
header('X-Powered-By: DEC Cookies API G4');

// CORS headers
if (isset($CONFIG['CORS_ALLOWED_ORIGINS'])) {
    $allowedOrigins = $CONFIG['CORS_ALLOWED_ORIGINS'];
    if (in_array('*', $allowedOrigins)) {
        header('Access-Control-Allow-Origin: *');
    } elseif (isset($_SERVER['HTTP_ORIGIN']) && in_array($_SERVER['HTTP_ORIGIN'], $allowedOrigins)) {
        header('Access-Control-Allow-Origin: ' . $_SERVER['HTTP_ORIGIN']);
    }
}
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: X-API-Key, X-Group-ID, Content-Type, Authorization');
header('Access-Control-Max-Age: 86400');
header('Access-Control-Allow-Credentials: true');

// Forcer HTTPS si configuré
if (($CONFIG['REQUIRE_HTTPS'] ?? false) && empty($_SERVER['HTTPS'])) {
    http_response_code(403);
    echo json_encode([
        'error' => 'HTTPS requis',
        'timestamp' => date('c')
    ]);
    exit;
}

// Gérer les requêtes OPTIONS (CORS preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ===== VÉRIFICATION VERSION CLIENT =====
if (isset($CONFIG['MIN_CLIENT_VERSION'])) {
    $clientVersion = $_SERVER['HTTP_X_CLIENT_VERSION'] ?? '1.0.0';
    if (version_compare($clientVersion, $CONFIG['MIN_CLIENT_VERSION'], '<')) {
        http_response_code(426);
        echo json_encode([
            'error' => 'Version client obsolète',
            'minimum_version' => $CONFIG['MIN_CLIENT_VERSION'],
            'current_version' => $clientVersion,
            'timestamp' => date('c')
        ]);
        exit;
    }
}

// ===== LIMITATION DE DÉBIT =====
if ($CONFIG['RATE_LIMIT_ENABLED'] ?? true) {
    require_once __DIR__ . '/rate_limiter.php';
    $rateLimiter = new RateLimiter($CONFIG);
    if (!$rateLimiter->checkLimit($_SERVER['REMOTE_ADDR'])) {
        http_response_code(429);
        header('Retry-After: ' . ($CONFIG['RATE_LIMIT_PERIOD'] ?? 60));
        echo json_encode([
            'error' => 'Trop de requêtes',
            'limit' => $CONFIG['RATE_LIMIT_REQUESTS'] ?? 100,
            'period' => $CONFIG['RATE_LIMIT_PERIOD'] ?? 60,
            'timestamp' => date('c')
        ]);
        exit;
    }
}

// ===== INITIALISATION DU STOCKAGE =====
$storageDir = $CONFIG['STORAGE_DIR'];
if (!is_dir($storageDir)) {
    if (!mkdir($storageDir, $CONFIG['DIR_PERMISSIONS'] ?? 0755, true)) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Erreur initialisation stockage',
            'timestamp' => date('c')
        ]);
        exit;
    }
}

// Créer les sous-dossiers nécessaires
$subDirs = ['sessions', 'backups', 'logs', 'temp', 'stats'];
foreach ($subDirs as $subDir) {
    $dir = $storageDir . $subDir . '/';
    if (!is_dir($dir)) {
        mkdir($dir, $CONFIG['DIR_PERMISSIONS'] ?? 0755, true);
    }
}

// ===== ROUTEUR =====
$method = $_SERVER['REQUEST_METHOD'];
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$action = getActionFromPath($path);

try {
    // Journal de la requête
    logRequest($CONFIG, $method, $path, $_SERVER['REMOTE_ADDR']);
    
    switch ($action) {
        case 'test':
            handleTest($CONFIG);
            break;
            
        case 'upload':
            if ($method !== 'POST') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleUpload($CONFIG);
            break;
            
        case 'download':
            if ($method !== 'GET') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleDownload($CONFIG);
            break;
            
        case 'stats':
            if ($method !== 'GET') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleStats($CONFIG);
            break;
            
        case 'delete':
            if ($method !== 'POST') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleDelete($CONFIG);
            break;
            
        case 'list':
            if ($method !== 'GET') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleList($CONFIG);
            break;
            
        case 'backup':
            if ($method !== 'GET') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleBackup($CONFIG);
            break;
            
        case 'cleanup':
            if ($method !== 'POST') {
                http_response_code(405);
                echo json_encode(['error' => 'Méthode non autorisée']);
                break;
            }
            handleCleanup($CONFIG);
            break;
            
        default:
            http_response_code(404);
            echo json_encode([
                'error' => 'Endpoint non trouvé',
                'available_endpoints' => [
                    'GET /test',
                    'POST /upload',
                    'GET /download',
                    'GET /stats',
                    'POST /delete',
                    'GET /list',
                    'GET /backup',
                    'POST /cleanup'
                ],
                'timestamp' => date('c')
            ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    $errorMessage = $CONFIG['DEBUG_MODE'] ?? false ? $e->getMessage() : 'Erreur serveur interne';
    
    echo json_encode([
        'error' => $errorMessage,
        'code' => $e->getCode(),
        'timestamp' => date('c')
    ]);
    
    logMessage($CONFIG, 'ERROR', $e->getMessage() . ' - ' . $e->getTraceAsString());
    
    // Notification par email si configuré
    if ($CONFIG['EMAIL_NOTIFICATIONS'] ?? false && $CONFIG['ALERT_ON_ERROR'] ?? false) {
        sendErrorEmail($CONFIG, $e);
    }
}

// ===== FONCTIONS (restent les mêmes que précédemment) =====
// ... [Les fonctions handleTest, handleUpload, etc. restent identiques] ...

// ===== NOUVELLES FONCTIONS =====
function handleBackup($config) {
    authenticate($config);
    
    $backupDir = $config['BACKUP_DIR'] ?? $config['STORAGE_DIR'] . 'backups/';
    if (!is_dir($backupDir)) {
        mkdir($backupDir, $config['DIR_PERMISSIONS'] ?? 0755, true);
    }
    
    $backupFile = $backupDir . 'backup_' . date('Y-m-d_His') . '.zip';
    
    // Créer une archive ZIP
    $zip = new ZipArchive();
    if ($zip->open($backupFile, ZipArchive::CREATE) === TRUE) {
        // Ajouter les sessions
        $sessionDir = $config['STORAGE_DIR'] . 'sessions/';
        $sessionFiles = glob($sessionDir . '*.json');
        
        foreach ($sessionFiles as $file) {
            $zip->addFile($file, 'sessions/' . basename($file));
        }
        
        // Ajouter l'index
        $indexFile = $config['STORAGE_DIR'] . 'index.json';
        if (file_exists($indexFile)) {
            $zip->addFile($indexFile, 'index.json');
        }
        
        // Ajouter les stats
        $statsDir = $config['STORAGE_DIR'] . 'stats/';
        if (is_dir($statsDir)) {
            $statsFiles = glob($statsDir . '*.json');
            foreach ($statsFiles as $file) {
                $zip->addFile($file, 'stats/' . basename($file));
            }
        }
        
        $zip->close();
        
        echo json_encode([
            'success' => true,
            'backup_file' => basename($backupFile),
            'file_size' => filesize($backupFile),
            'file_size_formatted' => formatBytes(filesize($backupFile)),
            'timestamp' => date('c'),
            'message' => 'Backup créé avec succès'
        ]);
        
        logMessage($config, 'BACKUP', 'Backup créé: ' . basename($backupFile));
        
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Impossible de créer le backup']);
    }
}

function handleCleanup($config) {
    authenticate($config);
    
    $days = intval($_POST['days'] ?? $config['SESSION_EXPIRE_DAYS'] ?? 30);
    $deletedCount = 0;
    
    $sessionDir = $config['STORAGE_DIR'] . 'sessions/';
    $files = glob($sessionDir . '*.json');
    
    $cutoff = time() - ($days * 24 * 60 * 60);
    
    foreach ($files as $file) {
        $fileTime = filemtime($file);
        if ($fileTime < $cutoff) {
            $sessionId = basename($file, '.json');
            if (unlink($file)) {
                removeFromIndex($config, $sessionId);
                $deletedCount++;
            }
        }
    }
    
    // Nettoyer les vieux logs
    if ($config['LOG_RETENTION_DAYS'] ?? false) {
        $logDir = $config['STORAGE_DIR'] . 'logs/';
        $logFiles = glob($logDir . '*.log');
        $logCutoff = time() - ($config['LOG_RETENTION_DAYS'] * 24 * 60 * 60);
        
        foreach ($logFiles as $logFile) {
            if (filemtime($logFile) < $logCutoff) {
                unlink($logFile);
            }
        }
    }
    
    echo json_encode([
        'success' => true,
        'deleted_sessions' => $deletedCount,
        'days' => $days,
        'timestamp' => date('c'),
        'message' => 'Nettoyage terminé'
    ]);
    
    logMessage($config, 'CLEANUP', "Nettoyage: $deletedCount sessions supprimées (> $days jours)");
}

function logRequest($config, $method, $path, $ip) {
    if (!$config['ENABLE_LOGS']) return;
    
    $logDir = $config['STORAGE_DIR'] . 'logs/';
    $logFile = $logDir . 'requests_' . date('Y-m-d') . '.log';
    
    $logEntry = sprintf(
        "[%s] %s %s - IP: %s - UA: %s\n",
        date('Y-m-d H:i:s'),
        $method,
        $path,
        $ip,
        substr($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown', 0, 100)
    );
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

function sendErrorEmail($config, $exception) {
    $to = $config['ADMIN_EMAIL'];
    $subject = 'ERREUR API Cookies G4';
    $message = sprintf(
        "Une erreur est survenue sur l'API Cookies G4:\n\n" .
        "Date: %s\n" .
        "IP: %s\n" .
        "URL: %s\n" .
        "Erreur: %s\n" .
        "Code: %d\n" .
        "Fichier: %s\n" .
        "Ligne: %d\n\n" .
        "Trace:\n%s",
        date('Y-m-d H:i:s'),
        $_SERVER['REMOTE_ADDR'],
        $_SERVER['REQUEST_URI'],
        $exception->getMessage(),
        $exception->getCode(),
        $exception->getFile(),
        $exception->getLine(),
        $exception->getTraceAsString()
    );
    
    $headers = 'From: ' . $config['ADMIN_EMAIL'] . "\r\n" .
               'X-Mailer: PHP/' . phpversion();
    
    mail($to, $subject, $message, $headers);
}
?>